/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package chart;

import rdc.RDCSample;

import com.inet.report.Chart2;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.RDC;
import com.inet.report.ReportException;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;
import com.inet.report.chart.ChartTitle;
import com.inet.report.chart.dataset.ForEachRecordDataset;
import com.inet.report.chart.format.FixedNumberFormat;
import com.inet.report.chart.plot.PieLegendLayout;
import com.inet.report.chart.plot.PiePlot;
import com.inet.report.chart.plot.PieStyle;

/**
 * This sample demonstrates how you can create a pie chart and use this properties.
 */
public class PieSample extends RDCSample {

    /**
     * The value column name.
     */
    private static String value = "Value";

    /**
     * Creates an empty engine, adds a chart and sets the data.
     * @param exportFmt the export format set by super class.
     * @return returns the created engine.
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //Creates a new empty engine.
            Engine engine = RDC.createEmptyEngine( exportFmt );
            //Creates a database column used by chart.
            TableSource tableSource = engine.getDatabaseTables().getDatasource( 0 ).createTableSource( "sample" );
            tableSource.addColumn( value, Field.NUMBER );

            //Gets the first section of report header.
            Section section = engine.getArea( Engine.REPORT_HEADER ).getSection( 0 );

            //Creates a new pie chart in the section.
            Chart2 pieChart = section.addChart2( PieStyle.PIE3D, 0, 0, 10000, 10000 );

            //Creates a data source of chart.
            PiePlot plot = (PiePlot)pieChart.getPlot();
            //This chart should shows each data record as a item.
            ForEachRecordDataset dataset = new ForEachRecordDataset( pieChart );
            //Adds the value field. The summary operation is irrelevant by this kind of dataset.
            DatabaseField valueField = tableSource.getDatabaseField( value );
            dataset.addDataField( SummaryField.NO_SUMMARY_OPERATION, valueField, null, 0 );
            plot.setDataset( dataset );

            //Gets the header title of chart.
            ChartTitle headerTitle = pieChart.getHeaderTitle();
            //Sets the empty label instead of automatic generated label.
            headerTitle.setShowAutoTitle( false );
            headerTitle.setTitle( "" );

            //Sets the start rotation angle of pie.
            plot.setRotationAngle( 45.0 );
            //Sets show label and value of each pie item.
            plot.setShowLabel( true );
            plot.setShowValue( true );
            //Expands the pie section number 0.
            plot.setSectionIndexes( new int[] { 0 } );
            //Sets the expanded gap.
            plot.setExpandPercent( 0.2 );

            //Sets the item label background color.
            plot.setItemLabelBackColor( RDC.COLOR_SILVER );
            //Sets the item label color.
            plot.setItemLabelColor( RDC.COLOR_BLACK );
            //Sets the item label gap between label and pie.
            plot.setItemLabelGap( 0.01 );
            //Sets the item label outline color.
            plot.setItemLabelOutlineColor( RDC.COLOR_GRAY );

            //Sets the percent number format.
            FixedNumberFormat format = new FixedNumberFormat( FixedNumberFormat.FORMAT_PERCENT );
            format.setNumberOfDecimalPlaces( 0 );
            plot.setItemLabelFormat( format );

            //Sets the pie transparency.
            plot.setForegroundAlpha( 0.5f );
            //Sets the pie outline color.
            plot.setOutlineColor( RDC.COLOR_BLACK );
            //Sets the pie outline width.
            plot.setOutlineWidth( 20 );

            //Sets the legend shows record number and percent value.
            plot.setLegendLayout( PieLegendLayout.SHOW_LABEL_AND_PERCENTAGE );

            //Sets the report data. 
            setData( engine );
            return engine;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Sets the report data. This sets the data column used by chart.
     * @param engine the current engine
     * @throws ReportException if engine is not initialized or finished.
     */
    private static void setData( Engine engine ) throws ReportException {
        String[] columns = { value };
        Object[][] data = new Object[4][];

        data[0] = new Object[1];
        data[0][0] = Integer.valueOf( 1 );

        data[1] = new Object[1];
        data[1][0] = Integer.valueOf( 2 );

        data[2] = new Object[1];
        data[2][0] = Integer.valueOf( 5 );

        data[3] = new Object[1];
        data[3][0] = Integer.valueOf( 4 );

        engine.setData( columns, data );
    }

    /**
     * Starts this sample.
     * @param argv nothing
     */
    public static void main( String[] argv ) {
        new PieSample().initUI();
    }
}
